import { FreeDatas2HTML, Selector } from "../src/FreeDatas2HTML";
const errors=require("../src/errors.js");
const fixtures=require("./fixtures.js");

describe("Test des sélecteurs de données", () =>
{
    let converter: FreeDatas2HTML;
    let selector: Selector;
    let selectElement : HTMLSelectElement;
    
    beforeEach( async () =>
    {
        document.body.insertAdjacentHTML("afterbegin", fixtures.datasViewEltHTML);
        converter=new FreeDatas2HTML("CSV");
        converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1.csv" });
        converter.datasViewElt={  id:"datas" };
        await converter.run();
    });

    afterEach( () =>
    {
        document.body.removeChild(document.getElementById("fixture"));
    });

    describe("Test des données d'initialisation.",  () =>
    {
        it("Doit générer une erreur, si initialisé sans avoir au préalable chargé des données.", async () =>
        {
            // Convertisseur non lancé :
            converter=new FreeDatas2HTML("CSV");
            expect(() => { return  new Selector(converter, 0, { id:"selector1" }); }).toThrowError(errors.filterNeedDatas);
            // Note : les parseurs vont générer une erreur en amont s'ils ne trouvent pas de noms de champs
            // Par contre, ils acceptent de ne pas trouver de données :
            converter.parser.datas2Parse="Z (numéro atomique),Élément,Symbole,Famille,Abondance des éléments dans la croûte terrestre (μg/k)";
            await converter.run();
            expect(() => { return  new Selector(converter, 0, { id:"selector1" }); }).toThrowError(errors.filterNeedDatas);
        });
        
         it("Doit générer une erreur, si le numéro de champ fourni n'existe pas dans les données.", () =>
        {
            expect(() => { return  new Selector(converter, 9, { id:"selector1" }); }).toThrowError(errors.selectorFieldNotFound);
            expect(() => { return  new Selector(converter, -1, { id:"selector1" }); }).toThrowError(errors.selectorFieldNotFound);
            expect(() => { return  new Selector(converter, 1.1, { id:"selector1" }); }).toThrowError(errors.selectorFieldNotFound);
        });

         it("Doit générer une erreur, si aucune donnée n'a été trouvée dans le champ du filtre.", async () =>
        {
            converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datasEmptyField.csv" });
            await converter.run();
            expect(() => { return  new Selector(converter, 3, { id:"selector1" });}).toThrowError(errors.selectorFieldIsEmpty);
        });
           
        it("Si un séparateur vide est fourni pour un filtre, il doit être ignoré.", () =>
        { 
            selector=new Selector(converter, 0, { id:"selector1" }, "");
            expect(selector.separator).toBeUndefined();
        });

        it("Si toutes les paramètres sont valides, ils doivent être acceptés et les informations correctement récupérées.", () =>
        {
            expect(() => { selector=new Selector(converter, 3, { id:"selector1" }, ","); return  true; }).not.toThrowError();
            expect(selector.datasFieldNb).toEqual(3);
            expect(selector.datasViewElt).toEqual({ id:"selector1", eltDOM:document.getElementById("selector1") });
            expect(selector.separator).toEqual(",");
            expect(selector.name).toEqual("Famille");
            expect(selector.values).toEqual(["Actinide","Gaz noble","gaz rare","Halogène","Indéfinie","Lanthanide","Métal alcalin","Métal alcalino-terreux","Métal de transition","Métal pauvre","Métalloïde","Non-métal"]);
        });

        it("Si des valeurs vides sont présentes dans un champ utilisé pour un filtre, elles doivent être ignorées.", async () =>
        {
            converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1-emtyinfield.csv" });
            await converter.run();
            selector=new Selector(converter, 3, { id:"selector1" }, "");
            expect(selector.values).toEqual(["Actinide","Gaz noble","gaz rare","Halogène","Indéfinie","Lanthanide","Métal alcalin","Métal alcalino-terreux","Métal de transition","Métal pauvre","Métalloïde","Non-métal"]);
        });
        
        it("Si des espaces entourent certaines valeurs pour ce champ, ils doivent être supprimés.", async () =>
        {
            converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datasNeedTrim.csv" });
            await converter.run();
            selector=new Selector(converter, 3, { id:"selector1" }, "");
            expect(selector.values).toEqual(["Gaz noble","Métal alcalin","Métal alcalino-terreux","Métalloïde","Non-métal"]);
        });

        it("Si un séparateur est fourni, les valeurs distinctes extraites de ce champ doivent le prendre en compte.", async () =>
        {
            converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1+tagsfield.csv" });
            await converter.run();
            selector=new Selector(converter, 5, { id:"selector1" }, "|");
            selector.filter2HTML(); 
            expect(selector.values).toEqual(["Exemple0","Exemple1","Exemple2","Exemple3","Exemple4","Exemple5","Exemple6","Exemple7","Exemple8","Exemple9","Exemple10"]);
        });
        
        it("Si une fonction spécifique est fournie pour le champ utilisé pour ce filtre, elle doit être prise en compte.",  () =>
        {
            const mySort=(a: any, b: any, order: "asc"|"desc"="asc") =>
            {
                const values=[ "> 100000", "> 1 et < 100 000", "≤ 1",  "Traces", "Inexistant"];
                if(order === "desc")
                    values.reverse();
                if(values.indexOf(a) > values.indexOf(b))
                    return -1;
                else if(values.indexOf(a) < values.indexOf(b))
                    return 1;
                else
                    return 0;
            };            
            converter.datasSortingFunctions=[{ datasFieldNb: 4, sort:mySort }];
            selector=new Selector(converter, 4, { id:"selector1" });
            expect(selector.values).toEqual(["Inexistant","Traces","≤ 1","> 1 et < 100 000","> 100000"]);
        });
    });

    describe("Création des sélecteurs.", () =>
    {
        beforeEach( async () =>
        {
            selector=new Selector(converter, 3, { id:"selector1" });
        });
        
        it("Doit générer un élement <select> listant les valeurs distinctes du champ spécifié, classées dans le bon ordre.",  () =>
        {
            selector.filter2HTML();            
            expect(document.getElementById("selector1").innerHTML).toEqual(fixtures.selector1HTML);
            selector=new Selector(converter, 4, { id:"selector2" });
            selector.filter2HTML();
            expect(document.getElementById("selector2").innerHTML).toEqual(fixtures.selector2HTML);
        });
        
        it("Doit prendre en compte l'éventuel label fourni pour le SELECT.",  () =>
        {
            selector=new Selector(converter, 4, { id:"selector2" });
            selector.filter2HTML("Abondance des éléments");
            expect(document.getElementById("selector2").innerHTML).toEqual(fixtures.selector2HTMLWithLabel);
        });
        
        it("Doit prendre en compte la possibilité de sélectionner plusieurs valeurs.",  () =>
        {
            selector=new Selector(converter, 4, { id:"selector2" });
            selector.isMultiple=true;
            selector.filter2HTML();
            expect(document.getElementById("selector2").innerHTML).toEqual(fixtures.selector2HTMLWithMultiple);
            //  le 'multiple' ajouté est transformé en 'multiple=""' par le navigateur.
        });
    });
           
    describe("Manipulation des sélecteurs et filtre des données.", () =>
    {
        beforeEach( async () =>
        {
            selector=new Selector(converter, 3, { id:"selector1" }); // filtre sur le champ "famille"
            selector.filter2HTML();
            selectElement=document.getElementById("freeDatas2HTML_selector1") as HTMLSelectElement;
        });

        it("La manipulation d'un sélecteur doit enregistrer la valeur sélectionnée et appeler la fonction actualisant l'affichage.",  () =>
        {
            spyOn(converter, "refreshView");

           selectElement.value="4";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues[0]).toEqual(3);
            expect(converter.refreshView).toHaveBeenCalledTimes(1);

            // Pas de choix multiples, donc seul le dernier choix est gardé :
            selectElement.value="2";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues[0]).toEqual(1);
            expect(converter.refreshView).toHaveBeenCalledTimes(2);

            selectElement.value="0"; // 0 = annulation de ce filtre, puisqu'aucune valeur choisie
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues.length).toEqual(0);
            expect(converter.refreshView).toHaveBeenCalledTimes(3);
        });

        it("Si plusieurs choix sont autorisés, toutes les valeurs doivent être enregistrées, l'affichage s'actualisant à chaque fois.",  () =>
        {
            spyOn(converter, "refreshView");
            
            selector=new Selector(converter, 3, { id:"selector1" }); 
            selector.isMultiple=true;
            selector.filter2HTML();
            selectElement=document.getElementById("freeDatas2HTML_"+selector.datasViewElt.id) as HTMLSelectElement;

            selectElement.options[4].selected=true;
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues[0]).toEqual(3);
            expect(converter.refreshView).toHaveBeenCalledTimes(1);

            // Le choix d'une nouvelle option ne supprime pas les précédents choix
            // Mais les valeurs sélectionnées sont toujours parcourues dans l'ordre du SELECT
            selectElement.options[3].selected=true;
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues[0]).toEqual(2);
            expect(selector.selectedValues[1]).toEqual(3);
            expect(converter.refreshView).toHaveBeenCalledTimes(2);

            // Le choix 0 annule les précédents choix
            selectElement.options[0].selected=true;
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.selectedValues.length).toEqual(0);
            expect(converter.refreshView).toHaveBeenCalledTimes(3);
        });
        
        it("Doit toujours retourner true si aucune des valeurs du filtre n'est sélectionnée.",  () =>
        {
            // Le filtre est sur 0 par défaut
            expect(selector.dataIsOk({ "nom"	: "oui" })).toBeTrue();
            // Même comportement après un retour :
            selectElement.value="2";
            selectElement.dispatchEvent(new Event("change"));
            selectElement.value="0";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "nom"	: "oui" })).toBeTrue();
        });
        
        it("Doit retourner false, si la donnée testée ne possède pas le champ sur lequel les données sont filtrées.",  () =>
        {
            selectElement.value="2";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "nom"	: "rémi sans famille" })).toBeFalse();
        });

        it("Doit générer une erreur si la valeur sélectionnée n'est pas trouvée dans la liste des valeurs connues.", () =>
        {
            selectElement.innerHTML=fixtures.selectorHTMLWithFakeItem;
            selectElement.value="13";
            selectElement.dispatchEvent(new Event("change"));
            expect(() => { selector.dataIsOk({ "Famille"	: "nombreuse" }); }).toThrowError(errors.selectorSelectedIndexNotFound);
        });
        
        it("Doit retourner false, si une donnée testée ne correspond pas à la valeur sélectionnée pour le filtre.",  async () =>
        {
            selectElement.value="4"; // = Halogène
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Famille": "Hallo Eugène !" })).toBeFalse();
        });
        
        it("Doit retourner true, si une donnée testée correspond à la valeur sélectionnée pour ce filtre.",  () =>
        {
            selectElement.value="4";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Famille": "Halogène" })).toBeTrue();
            // Y  compris si entouré d'espaces :
            expect(selector.dataIsOk({ "Famille": " Halogène " })).toBeTrue();
        });
        
        it("Doit retourner false, si une donnée testée ne correspond à aucune des valeurs sélectionnées dans un filtre multichoix.",  async () =>
        {
            selector=new Selector(converter, 3, { id:"selector1" }); 
            selector.isMultiple=true;
            selector.filter2HTML();
            selectElement=document.getElementById("freeDatas2HTML_"+selector.datasViewElt.id) as HTMLSelectElement;
            selectElement.options[3].selected=true;// sélection = gaz rare
            selectElement.options[4].selected=true; // sélection = Halogène
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Famille": "Hallo Eugène !" })).toBeFalse();
        });
        
        it("Doit retourner true, si une donnée testée correspond à au moins une des valeurs sélectionnées dans un filtre multichoix.",  () =>
        {
            selector=new Selector(converter, 3, { id:"selector1" }); 
            selector.isMultiple=true;
            selector.filter2HTML();
            selectElement=document.getElementById("freeDatas2HTML_"+selector.datasViewElt.id) as HTMLSelectElement;
            selectElement.options[3].selected=true;// sélection = gaz rare
            selectElement.options[4].selected=true; // sélection = Halogène
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Famille": "Halogène" })).toBeTrue();
            // Y  compris si entouré d'espaces :
            expect(selector.dataIsOk({ "Famille": " gaz rare " })).toBeTrue();
        });
    });
    
    describe("Manipulation des sélecteurs avec séparateur.", () =>
    {
        beforeEach( async () =>
        {
            converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1+tagsfield.csv" });
            await converter.run();
            selector=new Selector(converter, 5, { id:"selector1" }, "|"); // filtre sur le champ "Étiquettes"
            selector.filter2HTML();
            selectElement=document.getElementById("freeDatas2HTML_selector1") as HTMLSelectElement;;
        });
        
        it("Doit retourner false, si la donnée testée ne possède pas le champ sur lequel les données sont filtrées.",  () =>
        {
            selectElement.value="2";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "nom"	: "oui" })).toBeFalse();
        });
        
        it("Doit retourner false, si une donnée testée ne correspond pas à la valeur sélectionnée pour le filtre.",  () =>
        {
            selectElement.value="4"; // = Exemple3
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Étiquettes": "Mauvais exemple" })).toBeFalse();
        });
        
        it("Doit retourner true, si une donnée testée correspond pas à la valeur sélectionnée pour ce filtre.",  () =>
        {
            selectElement.value="4";
            selectElement.dispatchEvent(new Event("change"));
            expect(selector.dataIsOk({ "Étiquettes": "Exemple3" })).toBeTrue();
            // Même quand elle n'est pas seule :
            expect(selector.dataIsOk({ "Étiquettes": "Exemple3|Exemple1|Exemple9" })).toBeTrue();
            expect(selector.dataIsOk({ "Étiquettes": "Exemple0|Exemple3|Exemple2" })).toBeTrue();
            expect(selector.dataIsOk({ "Étiquettes": "Exemple0|Exemple4|Exemple3" })).toBeTrue();
            // Ou entourée d'espaces :
            expect(selector.dataIsOk({ "Étiquettes": "Exemple0|Exemple4| Exemple3 " })).toBeTrue();
            expect(selector.dataIsOk({ "Étiquettes": " Exemple3 " })).toBeTrue();
        });
    });
    
});