import { RemoteSource } from "../src/RemoteSource";
const errors=require("../src/errors.js");

describe("Tests des urls distantes", () =>
{
    let source: RemoteSource;
        
    beforeEach( () =>
    {
        source=new RemoteSource({ url:"http://localhost:8080/" });
    });

    it("Doit générer une erreur si l'url fournie est une chaîne vide.", () =>
    {
        expect(() => { return source.url=""; }).toThrowError(errors.remoteSourceNeedUrl);
        expect(() => { return source.url="  "; }).toThrowError(errors.remoteSourceNeedUrl);
    });
    
    it("Doit générer une erreur si l'url fournie n'utilise pas un des protocoles autorisés.", () =>
    {
        expect(() => { return source.url="htp://localhost:8080/"; }).toThrowError(errors.remoteSourceUrlFail);
        expect(() => { return source.url="ftp://localhost:8080/"; }).toThrowError(errors.remoteSourceUrlFail);
    });
    
    it("Doit accepter une url valide", () =>
    {
        expect(() => { return source.url="http://localhost:8080/"; }).not.toThrowError();
        source.allowedUrlProtocol.push("ftp:");
        expect(() => { return source.url="ftp://localhost:8080/"; }).not.toThrowError();
    });

    it("Seuls les headers conformes doivent être retenus", () =>
    {
        source.headers=[ { key:"token", value:"1234" }, { key:"vide", value:"" }, { key:"Sec-id", value:"666" }, { key:"Proxy-name", value:"myProxy" }, { key:"Content-Length", value:"255" }, { key:"userName", value:"Toto" }];
        expect(source.headers).toEqual([ { key:"token", value:"1234" }, { key:"vide", value:"" }, { key:"userName", value:"Toto" }]);
    });

    it("Doit retourner la configuration correcte pour fetch.", () =>
    {
        let headers=new Headers(); 
        expect(source.getFetchSettings()).toEqual({ method: "GET", headers: headers, credentials: "omit" });

        source.headers=[ { key:"token", value:"1234" }, { key:"userName", value:"Toto" }];
        source.withCredentials=true;
        headers=new Headers();            
        headers.append("token", "1234");
        headers.append("userName", "Toto");
        expect(source.getFetchSettings()).toEqual({ method: "GET", headers: headers, credentials: "include" });
    });
});