const Papa=require("papaparse");
import { RemoteSourceSettings } from "../src/interfaces";
import { ParserForCSV as Parser } from "../src/ParserForCSV";
const errors=require("../src/errors.js");

describe("Tests du parseur de CSV", () =>
{
    let parser: Parser;
        
    beforeEach( () =>
    {
        parser=new Parser();
    });

    it("Doit avoir créé une instance du Parser et initialiser la ressource distante avec une url vide.",  () =>
    {
        expect(parser).toBeInstanceOf(Parser);
        expect(parser.datasRemoteSource.url).toEqual("");
    });

    it("Doit générer une erreur si la chaîne de données à parser est vide.", () =>
    {
        expect(() => { return parser.datas2Parse="" }).toThrowError(errors.parserNeedDatas);
        expect(() => { return parser.datas2Parse="  " }).toThrowError(errors.parserNeedDatas);
    });

    it("Doit accepter toute chaîne de caractères non vide pour les données à parser.", () =>
    {
        parser. datas2Parse="datas";
        expect(parser.datas2Parse).toEqual("datas");
    });
    
    it("Doit générer une erreur si le parseur est lancé sans source de données fournie.", async () =>
    {
        await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserNeedSource));
    });

    it("Si les données sont directement fournies, cela doit être pris en compte dans les options passées à Papa Parse.",  () =>
    {
        // Idéalement il faudrait tester les paramètres passés à PapaParse
        // Mais semble impossible à cause des fonctions callback ?
        spyOn(Papa, "parse");
        parser.datas2Parse="datas";
        parser.parse();
        expect(parser.privateOptions).toEqual(
        {
            header:true,
            download:false,
            downloadRequestHeaders:undefined, 
            skipEmptyLines:"greedy",
            withCredentials:undefined
        });
        expect(Papa.parse).toHaveBeenCalledTimes(1);
    }); 
    
    it("Si une ressource distante est fournie, cela doit être pris en compte dans les options passées à Papa Parse.", () =>
    {
        // Même remarque, test précédent
        spyOn(Papa, "parse");
        let myRemoteSource: RemoteSourceSettings={ url:"http://localhost:9876/datas/datas1.csv",  withCredentials:true };
        parser.setRemoteSource(myRemoteSource);
        parser.parse();
        expect(parser.privateOptions).toEqual(
        {
            header:true,
            download:true,
            downloadRequestHeaders:{},// passe une liste de headers, même vide 
            skipEmptyLines:"greedy",
            withCredentials:true
        });
        myRemoteSource={ url:"http://localhost:9876/datas/datas1.csv",  headers:[ { key:"test", value:"coucou"}, { key:"test2", value:"coucou2"}] ,  withCredentials:false };
        parser.setRemoteSource(myRemoteSource);;
        parser.parse();
        expect(parser.privateOptions).toEqual(
        {
            header:true,
            download:true,
            downloadRequestHeaders:{ test:"coucou", test2:"coucou2"}, 
            skipEmptyLines:"greedy",
            withCredentials:false
        });
        expect(Papa.parse).toHaveBeenCalledTimes(2);
    });

    it("Si les données à parser contiennent des noms de champ vide ou en doublon, ils doivent être ignorés et cela doit être notifié.", async () =>
    {
        parser.datas2Parse="field1;field2;field3;field3; ";
        await parser.parse();
        expect(parser.parseResults.fields).toEqual(["field1","field2","field3"]);
        expect(parser.parseResults.errors).toEqual([{ row:-1, message: errors.parserFieldNameFail}, { row:-1, message: errors.parserFieldNameFail}]);
    });

    it("Doit générer une erreur si aucun nom de champ trouvé.", async () =>
    {
        parser.setRemoteSource({ url:"http://localhost:9876/datas/nodatas.csv" });
        await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserFieldsNotFound));
    });

    // Pas de test du fonctionnement de "Papa Parse" lui-même, car il s'agit d'un module externe.
    it("Si le parseur a été appelé avec les données nécessaires, des résultats doivent être enregistrés.", async () =>
    {
        parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1.csv" });
        await parser.parse();
        expect(parser.parseResults).not.toBeUndefined();
        // Y compris si les données fournies sont bidon :
        parser=new Parser();
        parser.datas2Parse="datas";
        await parser.parse();
        expect(parser.parseResults).not.toBeUndefined();
        // Mais dans ce cas, des erreurs doivent être reportées :
        expect(parser.parseResults.errors.length).not.toEqual(0);
    });
});