import { errors, FreeDatas2HTML, SortingField, SortingFieldsStandAlone } from "../../src/extensions/SortingFieldsStandAlone";
const fixtures=require("../fixtures.js");

describe("Test des liens de classement (hors données).", () =>
{
    let converter: FreeDatas2HTML;
    let sortingFields: SortingFieldsStandAlone;
    
    beforeEach( async () =>
    {
        document.body.insertAdjacentHTML("afterbegin", fixtures.datasViewEltHTML);
        converter=new FreeDatas2HTML("CSV");
        converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1.csv" });
        converter.datasViewElt={  id:"datas" };
        await converter.run(); // parsage + 1er affichage des données
    });

    afterEach( () =>
    {
        document.body.removeChild(document.getElementById("fixture"));
    });

    describe("Test des données reçues pour configurer les liens de classement.",  () =>
    {
        it("Doit générer une erreur, si initialisé avant que les champs de données ne soient connus.", () =>
        {
            converter=new FreeDatas2HTML("CSV");
            // Pas lancé converter.run(), donc les données n'ont pas été parsées :
            expect(() => { return  new SortingFieldsStandAlone(converter, { id:"selector1" }); }).toThrowError(errors.sortingFieldNeedDatas);
        });

        it("Doit générer une erreur, si initialisé avec l'id d'un élément n'existant pas dans le DOM", () =>
        {
            expect(() => { return  new SortingFieldsStandAlone(converter, { id:"dontExist" }); }).toThrowError(errors.converterElementNotFound+"dontExist");
        });
        
       it("Si tous les paramètres sont ok, ils doivent être acceptés.", () =>
        {
            expect(() => { sortingFields=new SortingFieldsStandAlone(converter, { id:"selector1" }); }).not.toThrowError();
            expect(sortingFields.converter).toEqual(converter);
            expect(sortingFields.datasViewElt.id).toEqual("selector1");
        });
    });

    describe("Création et action des liens de classement", () =>
    {
        let sortingField1: SortingField, sortingField2: SortingField;
        
        beforeEach(() =>
        {
            sortingFields=new SortingFieldsStandAlone(converter, { id:"selector1" });
            sortingField1=new SortingField(converter, 0);
            sortingField2=new SortingField(converter, 2);
            sortingFields.datasSortingFields=[sortingField1,sortingField2];
        });

        it("Doit générer une erreur, si lancé avant d'avoir fourni au moins un champ de classement.", () =>
        {
            sortingFields=new SortingFieldsStandAlone(converter, { id:"selector1" });
            expect(() => { return  sortingFields.rend2HTML() }).toThrowError(errors.needSortingFields);
        });

        it("Doit générer une liste de liens correspondant aux champs fournis .",  () =>
        {
            sortingFields.rend2HTML();
            let expectedHTML="<span></span><ul><li>"+fixtures.sortingColumn1HTML+"</li><li>"+fixtures.sortingColumn2HTML+"</li></ul>";
            expect(document.getElementById("selector1").innerHTML).toEqual(expectedHTML);
            // Idem avec des paramètres d'affichage :
            sortingFields.rendSettings=
            {
                allBeginning: "<h4>#LABEL</h4><p>",
                allEnding: "</p>",
                fieldBeginning: "",
                fieldEnding: " | ",
            };
            expectedHTML="<h4>Classer les données :</h4><p>"+fixtures.sortingColumn1HTML+" | "+fixtures.sortingColumn2HTML+" | </p>";
            sortingFields.rend2HTML("Classer les données :");
            expect(document.getElementById("selector1").innerHTML).toEqual(expectedHTML);
        });
        
        it("Lorsqu'ils sont cliqués, les liens de classement doivent transmettre l'information au convertisseur + lui demander d'actualiser l'affichage.", () =>
        {
            sortingFields.rend2HTML();
            let getLinks=document.querySelectorAll("ul li a") as NodeListOf<HTMLElement>;
            
            spyOn(converter, "refreshView"); 
            getLinks[0].click();// tri ascendant 1er champ
            expect(sortingField1.converter.datasSortedField).toEqual(sortingField1);
            expect(sortingField1.converter.datasSortedField.order).toEqual("asc");
            expect(converter.refreshView).toHaveBeenCalledTimes(1);
            
            getLinks[1].click();// tri ascendant mais sur le second champ
            expect(sortingField2.converter.datasSortedField).toEqual(sortingField2);
            expect(sortingField2.converter.datasSortedField.order).toEqual("asc");
            expect(converter.refreshView).toHaveBeenCalledTimes(2);
            
            getLinks[0].click();// tri descendant sur le 1er champ
            expect(sortingField1.converter.datasSortedField).toEqual(sortingField1);
            expect(sortingField1.converter.datasSortedField.order).toEqual("desc");
            expect(converter.refreshView).toHaveBeenCalledTimes(3);

            getLinks[0].click();// de nouveau ascendant
            expect(sortingField1.converter.datasSortedField).toEqual(sortingField1);
            expect(sortingField1.converter.datasSortedField.order).toEqual("asc");
            expect(converter.refreshView).toHaveBeenCalledTimes(4);
        });
    }); 
});