import { Render } from "../src/Render";
const errors=require("../src/errors.js");

describe("Test du moteur de rendu HTML.", () =>
{
    let render: Render;
    const fields=[ "Z", "Élément", "Symbole", "Famille" ] ;
    // Les champs des différentes lignes ne sont pas forcément dans le même ordre et certains peuvent être absents, vides ou encore en trop...
    const datas=[{"Z":"1","Élément":"Hydrogène","Symbole":"H","Famille":"Non-métal"},{"Famille":"Gaz noble","Élément":"Hélium","Z":"2","Symbole":"He"},{"Champ ignoré":"Je me champ ignoré !", "Z":"3","Élément":"Lithium","Famille":"Métal alcalin","Symbole":"Li"},{"Z":"4","Élément":"Béryllium","Famille":"","Champ ignoré":"Je me champ ignoré !"}] ;
    
    beforeEach(() =>
    {
        render=new Render();
    });

    it("Doit générer une erreur, si une liste de champs vide lui est fournie.", () =>
    {
        expect(() => { return  render.fields=[]; } ).toThrowError(errors.renderNeedFields);
        expect(render.fields).toEqual([]);
    });
    
    it("Doit accepter toute liste de champs valide.", () =>
    {
        render=new Render();
        expect(() => { return  render.fields=[""]; }).not.toThrowError(); // pas normal, mais pas testé par cette classe
        expect(render.fields).toEqual([""]);
        expect(() => { return  render.fields=["je vois double", "je vois double"]; }).not.toThrowError(); // idem
        expect(render.fields).toEqual(["je vois double", "je vois double"]);
        expect(() => { return  render.fields=["je me sens seul"]; }).not.toThrowError();
        expect(render.fields).toEqual(["je me sens seul"]);
    });

    it("Doit générer une erreur, si lancé sans avoir fourni une liste des champs.", () =>
    {
        expect(() => { return  render.rend2HTML(); }).toThrowError(errors.renderNeedFields);
    });
    
    it("Ne doit pas générer d'erreur, si lancé avec une liste des champs, même s'il n'y a aucune donnée fournie.", () =>
    {
        render.fields=fields;
        expect(() => { return  render.rend2HTML(); }).not.toThrowError();
    });

    it("Doit retourner le code HTML créé en se basant sur la configuration par défaut, avec ou sans données à afficher.", () =>
    {
        render.fields=fields;
        let html=render.rend2HTML();
        expect(html).toEqual("<table><thead><tr><th>Z</th><th>Élément</th><th>Symbole</th><th>Famille</th></tr></thead><tbody></tbody></table>");
        render.datas=datas; 
        html=render.rend2HTML(); 
        expect(html).toEqual("<table><thead><tr><th>Z</th><th>Élément</th><th>Symbole</th><th>Famille</th></tr></thead><tbody><tr><td>1</td><td>Hydrogène</td><td>H</td><td>Non-métal</td></tr><tr><td>2</td><td>Hélium</td><td>He</td><td>Gaz noble</td></tr><tr><td>3</td><td>Lithium</td><td>Li</td><td>Métal alcalin</td></tr><tr><td>4</td><td>Béryllium</td><td></td><td></td></tr></tbody></table>");                        
    });
    
    it("Doit retourner un code HTML correspondant à la configuration fournie, avec ou sans données à afficher.", () =>
    {
        render.settings=
        {
            allBegining:"<div id=\"test\"><h4>Pas de noms de champs !</h4>",
            allEnding:"</div>",
            linesBegining:"<div id=\"myDatas\">",
            linesEnding:"</div>",
            lineBegining:"<ul>",
            lineEnding:"</ul>",
            dataDisplaying:"<li><b>#FIELDNAME :</b> #VALUE</li>",
        };
        render.fields=fields;
        let html=render.rend2HTML();
        expect(html).toEqual("<div id=\"test\"><h4>Pas de noms de champs !</h4><div id=\"myDatas\"></div></div>");
        // Avec des données :
        render.datas=datas;
        html=render.rend2HTML(); 
        expect(html).toEqual("<div id=\"test\"><h4>Pas de noms de champs !</h4><div id=\"myDatas\"><ul><li><b>Z :</b> 1</li><li><b>Élément :</b> Hydrogène</li><li><b>Symbole :</b> H</li><li><b>Famille :</b> Non-métal</li></ul><ul><li><b>Z :</b> 2</li><li><b>Élément :</b> Hélium</li><li><b>Symbole :</b> He</li><li><b>Famille :</b> Gaz noble</li></ul><ul><li><b>Z :</b> 3</li><li><b>Élément :</b> Lithium</li><li><b>Symbole :</b> Li</li><li><b>Famille :</b> Métal alcalin</li></ul><ul><li><b>Z :</b> 4</li><li><b>Élément :</b> Béryllium</li><li><b>Symbole :</b> </li><li><b>Famille :</b> </li></ul></div></div>"); 
    });
});