import { ParserForJSON as Parser } from "../src/ParserForJSON";
const errors=require("../src/errors.js");

describe("Tests du parseur de JSON", () =>
{
    let parser: Parser;
        
    beforeEach( () =>
    {
        parser=new Parser();
    });

    it("Doit avoir créé une instance du Parser et initialiser la ressource distante avec une url vide.",  () =>
    {
        expect(parser).toBeInstanceOf(Parser);
        expect(parser.datasRemoteSource.url).toEqual("");
    });

    it("Doit générer une erreur si la chaîne de données à parser est vide.", () =>
    {
        expect(() => { return parser.datas2Parse="" }).toThrowError(errors.parserNeedDatas);
        expect(() => { return parser.datas2Parse="  " }).toThrowError(errors.parserNeedDatas);
    });

    it("Doit accepter toute chaîne de caractères non vide pour les données à parser.", () =>
    {
        parser. datas2Parse="datas";
        expect(parser.datas2Parse).toEqual("datas");
    });

    it("Doit générer une erreur si le parseur est lancé sans source de données fournie.", async () =>
    {
        await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserNeedSource));
    });

    it("Si le parseur a été appelé avec des données correctes, des résultats doivent être enregistrés.", async () =>
    {
        parser.datas2Parse= `[{ "nom": "dugenoux"},{ "nom": "dupont"}]`;
        await parser.parse();
        expect(parser.parseResults).not.toBeUndefined();
    });

    describe("Accès à des données distantes.", () =>
    {
        it("Doit générer une erreur, si l'accès aux données distantes est défaillant.", async () =>
        {
            parser.setRemoteSource({ url:"http://localhost:9876/datas/posts.jso" }); // une seule lettre vous manque...
            await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserRemoteFail));
        });

        it("Si le parseur a été appelé avec une url fournissant des données correctes, des résultats doivent être enregistrés.", async () =>
        {
            parser.setRemoteSource({ url:"http://localhost:9876/datas/posts.json" });
            await parser.parse();
            expect(parser.parseResults).not.toBeUndefined();
        });

        it("Si des options de connexion distante sont fournies, elles doivent être utilisées.", async () =>
        {
            spyOn(window, "fetch").and.callThrough();
            parser.setRemoteSource({ url:"http://localhost:9876/datas/posts.json", headers: [{ key:"token", value:"1234" }, { key:"userName", value:"Toto" }], withCredentials:true });
            await parser.parse();
            const headers=new Headers();
            headers.append("token", "1234");
            headers.append("userName", "Toto");
            expect(window.fetch).toHaveBeenCalledWith("http://localhost:9876/datas/posts.json", { method: "GET", headers: headers, credentials: "include" });
        });
    });

    describe("Noms des champs et données fournies dans deux tableaux distincts.", () =>
    {
        it("Les espaces entourant les noms de champs doivent être supprimés.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: [" nom", "prénom ", " âge "], datas: [] });
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
        });
        
        it("Les noms de champs vides, en doublon ou autre qu'une chaîne de caractère doivent ête refusés et l'erreur reportée.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: ["nom", 24, "prénom", true,{ field:"champ"}, "âge", ["je suis un nom de champ"], " ", "nom"], datas: [] });
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
            expect(parser.parseResults.errors).toEqual([{ row:-1, message: errors.parserTypeError+"number" },{ row:-1, message: errors.parserTypeError+"boolean" }, { row:-1, message: errors.parserTypeError+"object" }, { row:-1, message: errors.parserTypeError+"object" },{ row:-1, message: errors.parserFieldNameFail}, { row:-1, message: errors.parserFieldNameFail}]);
        });

        it("Une erreur doit être générée si aucun nom de champ valide n'est trouvé.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: [{ field:"champ"}, " "], datas: [] });
            await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserFail));
        });
        
        it("Si une ligne contient un nombre de champs différents que celui attendu, l'erreur doit être reportée. Les champs en trop sont ignorés.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: ["nom","prénom", "âge"], datas: [["dugenoux","henri",25,"je me champ en trop"],["michu","mariette"]] });
            await parser.parse();
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux","prénom":"henri", "âge":"25"}, {nom:"michu","prénom":"mariette"}]);
            expect(parser.parseResults.errors).toEqual([{ row:0, message:errors.parserNumberOfFieldsFail}, { row:1, message:errors.parserNumberOfFieldsFail}]);
        });

        it("Si certaines des données fournies ont un type non accepté, elles doivent être ignorées et les erreurs doivent être reportées.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: ["nom","prénom", "âge"], datas: [["dugenoux",{ "prenom":"henri"},25],["michu","mariette",null]] });
            await parser.parse();
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux", "âge":"25"}, {nom:"michu","prénom":"mariette"}]);
            expect(parser.parseResults.errors[0]).toEqual({row:0,message:errors.parserTypeError+"object"});
            expect(parser.parseResults.errors[1]).toEqual({row:1,message:errors.parserTypeError+"object"});
        });
        
        it("Un enregistrement n'ayant aucune donnée valide sera ignoré. L'erreur est reportée.", async () =>
        {
           parser.datas2Parse=JSON.stringify({ fields: ["nom","prénom", "âge"], datas: [["dugenoux","henri",25],[null,{ "prenom":"mariette"},[58]]] });
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux","prénom":"henri", "âge":"25"}]);
            expect(parser.parseResults.errors[3]).toEqual({row:1,message:errors.parserLineWithoutDatas});
        });

         it("Si toutes les données fournies sont ok, on doit les retrouver en résultat et aucune erreur n'est reportée.", async () =>
        {
            parser.datas2Parse=JSON.stringify({ fields: ["nom","prénom", "âge"], datas: [["dugenoux","henri",25],["michu","mariette",58]] });          
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux","prénom":"henri", "âge":"25"}, {nom:"michu","prénom":"mariette", "âge":"58"}]);
            expect(parser.parseResults.errors.length).toEqual(0);
        });
   });

    describe("Données fournies sous forme de tableau d'objets.", () =>
    {                
        it("Les espaces entourant les noms de champs doivent être supprimés.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{"nom  ":"dugenoux","  prénom":"henri"," âge ":25},{nom:"michu","prénom":"mariette","âge":58}]);
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
        });

        it("Si certaines des données fournies ont un type non accepté, elles doivent être ignorées ainsi que leur attribut. Et les erreurs doivent être reportées.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{nom:"dugenoux","prénom":{"value":"henri"},"âge":25},{"âge":"58",nom:"michu","prénom":"mariette",pseudo:["madame Michu"]}]);
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom", "âge", "prénom"]);
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux", "âge":"25"}, {"âge":"58", nom:"michu", "prénom":"mariette"}]);
            expect(parser.parseResults.errors[0]).toEqual({row:0,message:errors.parserTypeError+"object"});
            expect(parser.parseResults.errors[1]).toEqual({row:1,message:errors.parserTypeError+"object"});
        });
        
        it("Si certaines des données fournies déclare plusieurs fois le même attribut, elles doivent être ignorées. Et les erreurs doivent être reportées.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{nom:"dugenoux","prénom":"henri","âge":25, "prénom ":"Henry"},{"âge":"58",nom:"michu","prénom":"mariette", " âge ":"48" }]);
            await parser.parse();
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux", "prénom":"henri","âge":"25"}, {"âge":"58", nom:"michu", "prénom":"mariette"}]);
            expect(parser.parseResults.errors[0]).toEqual({row:0,message:errors.parserFieldNameFail });
            expect(parser.parseResults.errors[1]).toEqual({row:1,message:errors.parserFieldNameFail });
        });
        
        it("Un enregistrement n'ayant aucune donnée valide doit être ignoré et cela doit être reporté.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{nom:["dugenoux"]},{nom:"michu","prénom":"mariette","âge":58}]);
            await parser.parse();
            expect(parser.parseResults.datas).toEqual([{ nom:"michu","prénom":"mariette","âge":"58"}]);
            expect(parser.parseResults.errors[1]).toEqual({row:0,message:errors.parserLineWithoutDatas }); // errors[0] signale l'erreur de type
        });

        it("Doit générer une erreur si aucun nom de champ n'est trouvé dans les données.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{" ":"dugenoux","  ":"henri"},{" ":"michu","  ":"   "}]);
            await expectAsync(parser.parse()).toBeRejectedWith(new Error(errors.parserFail));
        });

        it("Si toutes les données fournies sont ok, on doit les retrouver en résultat et aucune erreur n'est reportée.", async () =>
        {
            parser.datas2Parse=JSON.stringify([{nom:"dugenoux","prénom":"henri","âge":25},{nom:"michu","prénom":"mariette","âge":58}]);
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge"]);
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux","prénom":"henri", "âge":"25"}, {nom:"michu","prénom":"mariette", "âge":"58"}]);
            expect(parser.parseResults.errors.length).toEqual(0);
            // Tous les objets n'ont pas forcément les mêmes attributs, ni dans le même ordre
            parser.datas2Parse=JSON.stringify([{nom:"dugenoux","prénom":"henri","âge":"25"},{"âge":"58",nom:"michu",pseudo:"madame Michu"}]);
            await parser.parse();
            expect(parser.parseResults.fields).toEqual(["nom","prénom", "âge", "pseudo"]);
            expect(parser.parseResults.datas).toEqual([{nom:"dugenoux","prénom":"henri", "âge":"25"}, {"âge":"58", nom:"michu", pseudo:"madame Michu" }]);
            expect(parser.parseResults.errors.length).toEqual(0);
        });
   });
});