import { FreeDatas2HTML, Pagination } from "../src/FreeDatas2HTML";
const errors=require("../src/errors.js");
const fixtures=require("./fixtures.js");

describe("Test de la pagination.", () =>
{
    let converter: FreeDatas2HTML;
    let pagination: Pagination;
    
    beforeEach( async () =>
    {
        document.body.insertAdjacentHTML("afterbegin", fixtures.datasViewEltHTML);
        converter=new FreeDatas2HTML("CSV");
        converter.parser.setRemoteSource({ url:"http://localhost:9876/datas/datas1.csv" });
        converter.datasViewElt={ id:"datas" };
        await converter.run();
        pagination=new Pagination(converter, { id:"pages" }, "Page à afficher :");
    });

    afterEach( () =>
    {
        document.body.removeChild(document.getElementById("fixture"));
    });

    describe("Test des options de pagination.",  () =>
    {
        it("Doit retourner un booléen indiquant si un nombre est un entier positif ou non.", () =>
        {
           expect(Pagination.isPositiveInteger(-1)).toBeFalse(); 
           expect(Pagination.isPositiveInteger(1.25)).toBeFalse();
           expect(Pagination.isPositiveInteger(0)).toBeFalse();
           expect(Pagination.isPositiveInteger(1)).toBeTrue();
        });

        it("Doit générer une erreur si la pagination est initialisée sans données à traiter.", () =>
        {
            converter=new FreeDatas2HTML("CSV");
            expect(() => { return new Pagination(converter, { id:"pages" }); }).toThrowError(errors.paginationNeedDatas);
        });
                        
        it("Ne doit pas générer d'erreur si initialisé correctement", () =>
        {
            expect(() => { return new Pagination(converter, { id:"pages" }); }).not.toThrowError();
        });

        it("Doit générer une erreur si la pagination par défaut n'est pas un entier positif.", () =>
        {
            expect(() => { return pagination.selectedValue=0; }).toThrowError(errors.needPositiveInteger);
        });

        it("Doit accepter toute valeur de pagination par défaut valide, y compris \"undefined\".", () =>
        {
            expect(() => { return pagination.selectedValue=10; }).not.toThrowError();
            expect(pagination.selectedValue).toEqual(10);
            expect(() => { return pagination.selectedValue=undefined; }).not.toThrowError();
            expect(pagination.selectedValue).toBeUndefined();
        });

        it("Doit générer une erreur si les options de pagination sont initialisées avec un tableau de valeurs comptant moins de 2 valeurs distinctes.", () =>
        {
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [] }; }).toThrowError(errors.paginationNeedOptionsValues);
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10] }; }).toThrowError(errors.paginationNeedOptionsValues);
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,10] }; }).toThrowError(errors.paginationNeedOptionsValues);
        });
        
        it("Doit générer une erreur si au moins une des options de pagination fournies n'est pas un entier positif.",  () =>
        {
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [0,10,20] }; }).toThrowError(errors.needPositiveInteger);
        });
        
        it("Doit accepter toutes les options de pagination valides.", () =>
        {
            const domElement=document.getElementById("paginationOptions");
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [1,2] }; }).not.toThrowError();
            expect(pagination.options).toEqual({ displayElement: { id:"paginationOptions", eltDOM: domElement }, values: [1,2], name:"Pagination" });
            // Doublons ignorés mais anomalie non bloquante :
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,20,20] }; }).not.toThrowError(); 
            expect(pagination.options).toEqual({ displayElement: { id:"paginationOptions", eltDOM: domElement }, values: [10,20], name:"Pagination" });
            // L'ordre des valeurs proposées est libre :
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [100,20,50,10] }; }).not.toThrowError(); 
            expect(pagination.options).toEqual({ displayElement: { id:"paginationOptions", eltDOM: domElement }, values: [100,20,50,10], name:"Pagination" });
            // Undefined accepté :
            expect(() => { return pagination.options=undefined; }).not.toThrowError();
            expect(pagination.options).toBeUndefined();
            // Un libellé pour l'affichage peut aussi être fourni
            expect(() => { return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,50], name: "La pagination" } }).not.toThrowError();
            expect(pagination.options).toEqual({ displayElement: { id:"paginationOptions", eltDOM: domElement }, values: [10,50], name:"La pagination" });
        });

        it("Doit générer une erreur si la pagination par défaut ne fait pas partie des valeurs proposées en option.", () =>
        {
            expect(() => { pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,20,50] }; return pagination.selectedValue=15; }).toThrowError(errors.paginationNeedByfaultValueBeInOptions);
            // Idem dans l'autre sens :
            expect(() => { pagination.selectedValue=15; return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,20,50] }; }).toThrowError(errors.paginationNeedByfaultValueBeInOptions);
        });

        it("Doit accepter des valeurs cohérentes pour les options et la valeur par défaut.", () =>
        {
            expect(() => { pagination.options={ displayElement: { id:"paginationOptions" }, values: [20,50] }; return pagination.selectedValue=50; }).not.toThrowError();
            // Idem dans l'autre sens :
            expect(() => { pagination.selectedValue=20; return pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,20] }; }).not.toThrowError();
        });
    });

    describe("Création et action des sélecteurs liés à la pagination des données.", () =>
    {
        beforeEach( () =>
        {
            pagination.options={ displayElement: { id:"paginationOptions" }, values: [10,20,50,500] , name: "Choix de pagination :" };
            pagination.selectedValue=10;
            pagination.options2HTML();
            converter.pagination=pagination;
        });

        it("Doit générer une erreur, si lancé sans options de pagination.", () =>
        {
            pagination.options=undefined;
            expect(() => { return pagination.options2HTML(); }).toThrowError(errors.pagination2HTMLFail);
        });

        it("Si des options de pagination sont fournies, doit générer un élement <select> listant les valeurs possibles.", () =>
        {
            pagination.options2HTML();
            expect(document.getElementById("paginationOptions").innerHTML).toEqual(fixtures.selectorForPagination);
        });
        
        it("Si une valeur de pagination par défaut est connue, elle doit être sélectionnée dans la liste.", () =>
        {
            pagination.selectedValue=20;
            pagination.options2HTML();
            const selectElement=document.getElementById("freeDatas2HTMLPaginationSelector") as HTMLInputElement;
            expect(selectElement.value).toEqual("2");
        });
        
        it("La manipulation du sélecteur de pagination doit enregistrer la valeur sélectionnée et appeler la fonction actualisant l'affichage.", () =>
        {
            spyOn(converter, "refreshView");
            const selectElement=document.getElementById("freeDatas2HTMLPaginationSelector") as HTMLInputElement;
            selectElement.value="2";
            selectElement.dispatchEvent(new Event('change'));
            expect(pagination.selectedValue).toEqual(20);
            expect(converter.refreshView).toHaveBeenCalledTimes(1);
            selectElement.value="0";
            selectElement.dispatchEvent(new Event('change'));
            expect(pagination.selectedValue).toBeUndefined();
            expect(converter.refreshView).toHaveBeenCalledTimes(2);
        });

        it("S'il n'y a pas de pagination définie, le sélecteur de pages ne doit pas être proposé.", () =>
        {
            pagination.selectedValue=undefined;
            pagination.pages2HTML();
            const selectElement=document.getElementById("pages").innerHTML;
            expect(selectElement).toEqual("");
           });

        it("S'il n'y pa plus de données que le nombre de lignes par page, le sélecteur de pages ne doit pas être proposé.", () =>
        {
            pagination.selectedValue=500;
            pagination.pages2HTML();
            const selectElement=document.getElementById("pages").innerHTML;
            expect(selectElement).toEqual("");
        });

        it("S'il y a plus de données que le nombre de lignes autorisées par page, un <select> listant les pages doit être affiché.", () =>
        {
            pagination.pages2HTML();
            const selectElement=document.getElementById("pages").innerHTML;
            expect(selectElement).toEqual(fixtures.selectorForManyPages);
        });
        
        /*it("Après chaque actualisation de la liste de pages, la valeur sélectionnée est remise à 0.", () =>
        {
            pagination.pages2HTML();
            let selectElement=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            expect(pagination.pages.selectedValue).toEqual(1);
            pagination.pages.selectedValue=2;
            pagination.pages2HTML();
            selectElement=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            expect(pagination.pages.selectedValue).toEqual(1);
        });*/
        
        it("La manipulation du sélecteur de pages doit enregistrer la valeur sélectionnée et appeler la fonction actualisant l'affichage, tout en gardant la page choisie présélectionnée.", () =>
        {
            pagination.pages2HTML();
            spyOn(converter, "refreshView");
            const selectElement=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            selectElement.value="4"; 
            selectElement.dispatchEvent(new Event('change'));
            expect(pagination.pages.selectedValue).toEqual(4);
            expect(converter.refreshView).toHaveBeenCalledTimes(1);
            let selectElementAfter=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            expect(selectElementAfter.value).toEqual("4");
            selectElement.value="1";
            selectElement.dispatchEvent(new Event('change'));
            expect(pagination.pages.selectedValue).toEqual(1);
            expect(converter.refreshView).toHaveBeenCalledTimes(2);
            selectElementAfter=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            expect(selectElementAfter.value).toEqual("1");
        });

        it("Tout actualisation de l'affichage non généré par le choix d'une page, doit supprimer ce choix.", () =>
        {
            pagination.pages2HTML();
            const selectElement=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            selectElement.value="4"; 
            selectElement.dispatchEvent(new Event('change'));
            converter.refreshView();
            const selectElementAfter=document.getElementById("freeDatas2HTMLPagesSelector") as HTMLInputElement;
            expect(selectElementAfter.value).toEqual("1");
        });
        
    });
});